/*
 *  							PS/2 Keyer
 *
 *  Copyright (C) 2009  David Bern, W2LNX     W2LNX@ARRL.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 *  USA, or see <http://www.gnu.org/licenses/>.
 */

#include "device.h"
#include "configuration.h"
#include "constants.h"
#include "version.h"
#include "cw_paddle_keyboard.h"
#include "cw_paddle_mouse.h"
#include "ps2_keyboard.h"
#include "ps2_mouse.h"
#include "interrupts.h"
#include "main.h"
#include <assert.h>

/******************************************************************************/
/*																			  */
/*		This program emulates a PS/2 keyboard and a PS/2 mouse using a CW	  */
/*		keyer paddle.														  */
/*																			  */			
/******************************************************************************/

/* -------------------------------------------------------------------------- */

static void initialize(void);

static void routine_dispatcher(void);

static void blink_port_LEDs(void);

#if TESTING == 1
void test_routines(void);
#endif

/* -------------------------------------------------------------------------- */

/*
 * main routine
 */

void main(void)
{

	delay_ms(100);
	blink_port_LEDs();
	blink_port_LEDs();
#if TESTING == 1
	blink_status_led(2);
	output_low(MARK_PIN);	
#endif

	printf("\r\n");
	printf("%s\r\n", get_version());
	printf("%s\r\n", get_copyright_notice());
	printf("compiler version: %s\r\n", getenv("VERSION_STRING"));
	printf("device: %s\r\n", getenv("DEVICE"));		

	initialize();

	delay_ms(1000);

#if PS2_KEYER_KEYBOARD == 1
	demo_ps2_keyboard();
#endif
#if PS2_KEYER_MOUSE  == 1
	demo_ps2_mouse();
#endif
	delay_ms(1000);

#if TESTING == 1
	test_routines();
	sleep();
#endif

	routine_dispatcher();
}

/* -------------------------------------------------------------------------- */

/*
 * main initialization routine
 */

static void initialize(void)
{

	/* internal port B pull up resistors */
	port_b_pullups(0xFF);

#if PS2_KEYER_KEYBOARD == 1
	initialize_ps2_keyboard();
	initialize_cw_paddle_keyboard();
#endif

#if PS2_KEYER_MOUSE  == 1
	initialize_ps2_mouse();
	initialize_cw_paddle_mouse();
#endif

	intialize_interrupts();

	printf("hello...\r\n");
	printf("\r\n");
}

/* -------------------------------------------------------------------------- */

/*
 * this routine dispatches to the appropriate routine.  This routine is
 * specific for a CW paddle that is connected
 */

static void routine_dispatcher(void) 
{

	delay_ms(100);

	for (;;) {
#if PS2_KEYER_KEYBOARD == 1
		cw_paddle_keyboard();
#endif
#if PS2_KEYER_MOUSE == 1
		cw_paddle_mouse();
#endif
	}
}

/* -------------------------------------------------------------------------- */

/*
 * this routine returns the project name, its version number and 
 * when it was compiled
 */

char *get_version(void)
{
	static char str[ ] = PROJECT " " VERSION " (" __DATE__ " "__TIME__ ")";

	return str;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine returns the copyright notice
 */

char *get_copyright_notice(void)
{
	static char str[ ] =
		"copyright (C) 2009 David Bern, W2LNX     W2LNX@ARRL.NET";

	return str;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine returns the license notice
 */

char *get_license_notice(void)
{
	static char str[ ] =
		"This program is free software; you can redistribute it and/or\r"
		"modify it under the terms of the GNU General Public License\r"
		"as published by the Free Software Foundation; either version 2\r"
		"of the License, or (at your option) any later version.\r"
		"\r"
		"etc., etc.";

	return str;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine blinks all the port LEDs
 */

const unsigned int port_led[ ] = { PORT0_LED, PORT1_LED, PORT3_LED, PORT2_LED };
	
enum {
	port_led_size = sizeof(port_led)/sizeof(port_led[0])
};

static void blink_port_LEDs(void)
{
	unsigned int i;

	for (i = 0; i < port_led_size; i++) {
		output_high(port_led[i]);
	}
	delay_ms(250);
	
	for (i = 0; i < port_led_size; i++) {
		output_low(port_led[i]);
	}
	delay_ms(250);
}

/* -------------------------------------------------------------------------- */

/******************************************************************************/
/*																			  */
/*							test routines									  */
/*																			  */			
/******************************************************************************/

#if TESTING == 1
/*
 * select the test case
 */

void test_routines(void)
{
	
	delay_ms(100);
	printf("test_routines():\r\n");

	for (;;) {
#if PS2_KEYER_KEYBOARD == 1
		test_cw_paddle_keyboard();
#endif
#if PS2_KEYER_MOUSE == 1
		test_cw_paddle_mouse();
#endif
	}

	printf("test_routines() done\r\n");
	delay_ms(10);
}

/* -------------------------------------------------------------------------- */

/*
 * this routine blinks the status LED
 */

void blink_status_led(unsigned int count)
{
	unsigned int i;

	for (i = 0; i < count; i++) {
		output_high(STATUS_LED);
		delay_ms(250);
		output_low(STATUS_LED);
		delay_ms(250);
	}
}

/* -------------------------------------------------------------------------- */
#endif
